/*
 * Decompiled with CFR 0.152.
 */
package oracle.ideimpl.persistence;

import com.sleepycat.je.CheckpointConfig;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DatabaseNotFoundException;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.RunRecoveryException;
import com.sleepycat.je.SecondaryConfig;
import com.sleepycat.je.SecondaryDatabase;
import com.sleepycat.je.SecondaryMultiKeyCreator;
import com.sleepycat.je.log.LogException;
import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.logging.Level;
import net.jcip.annotations.GuardedBy;
import oracle.ide.performance.PerformanceLogger;
import oracle.ide.persistence.SecondaryKeyProvider;
import oracle.ideimpl.persistence.BerkeleyDBNameSpace;
import oracle.ideimpl.persistence.BerkeleyDBOperation;
import oracle.ideimpl.persistence.OutOfFileHandlesDialog;
import oracle.ideimpl.persistence.PersistenceLogger;
import oracle.ideimpl.persistence.SecondaryKeyCreatorAdapter;
import oracle.javatools.annotations.NotNull;
import oracle.javatools.annotations.Nullable;

final class BerkeleyDBEnvironment {
    static final boolean USE_DEFERRED_WRITE = true;
    static final Iterator<String> EMPTY_ITERATOR = new Iterator<String>(){

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public String next() {
            throw new NoSuchElementException("This iterator does not support element retrieval.");
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("This iterator does not support removal of elements.");
        }
    };
    @NotNull
    private static final EnvironmentConfig ENVIRONMENT_CONFIG = new EnvironmentConfig();
    @NotNull
    private static final DatabaseConfig DATABASE_CONFIG = new DatabaseConfig();
    @NotNull
    private final String location;
    @GuardedBy(value="this")
    @Nullable
    private Environment environment;
    @GuardedBy(value="this")
    @NotNull
    private final Map<String, BerkeleyDBNameSpace> namespaces = new HashMap<String, BerkeleyDBNameSpace>();

    @NotNull
    private static final void initConfig() {
        ENVIRONMENT_CONFIG.setAllowCreate(true);
        boolean useDefaultSize = true;
        long defaultSize = 0x500000L;
        long minimumSize = 98304L;
        try {
            String property = System.getProperty("ide.persistence.cache.percent");
            if (property != null) {
                int percent = Integer.parseInt(property);
                if (percent >= 0) {
                    ENVIRONMENT_CONFIG.setCachePercent(percent);
                    useDefaultSize = false;
                    PersistenceLogger.getLogger().config("Persistence cache size is " + percent + "% of total memory");
                }
            } else {
                long size;
                property = System.getProperty("ide.persistence.cache.size");
                if (property != null && (size = Long.parseLong(property)) >= 98304L) {
                    ENVIRONMENT_CONFIG.setCacheSize(size);
                    useDefaultSize = false;
                    PersistenceLogger.getLogger().config("Persistence cache size is " + size + " bytes");
                }
            }
        }
        catch (NumberFormatException e) {
        }
        catch (IllegalArgumentException e) {
            // empty catch block
        }
        if (useDefaultSize) {
            ENVIRONMENT_CONFIG.setCacheSize(0x500000L);
            PersistenceLogger.getLogger().config("Using default persistence cache size (5242880 bytes)");
        }
        ENVIRONMENT_CONFIG.setConfigParam("je.env.runCleaner", "false");
        ENVIRONMENT_CONFIG.setConfigParam("je.env.runCheckpointer", "false");
        ENVIRONMENT_CONFIG.setConfigParam("je.env.runINCompressor", "false");
        ENVIRONMENT_CONFIG.setConfigParam("je.log.fileMax", "1000000");
        ENVIRONMENT_CONFIG.setConfigParam("je.cleaner.minAge", "1");
        ENVIRONMENT_CONFIG.setConfigParam("je.cleaner.minUtilization", "75");
        ENVIRONMENT_CONFIG.setConfigParam("je.cleaner.minFileUtilization", "30");
        ENVIRONMENT_CONFIG.setConfigParam("je.log.fileCacheSize", "3");
        DATABASE_CONFIG.setDeferredWrite(true);
        DATABASE_CONFIG.setAllowCreate(true);
    }

    public BerkeleyDBEnvironment(String location) {
        this.location = location;
    }

    @NotNull
    public String getLocation() {
        return this.location;
    }

    public synchronized void close() {
        for (BerkeleyDBNameSpace namespace : this.namespaces.values()) {
            try {
                namespace.closeDatabase();
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to close database " + namespace, e);
            }
        }
        this.namespaces.clear();
        try {
            this.closeEnvironment();
        }
        catch (DatabaseException e) {
            PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to close environment " + this, e);
        }
    }

    @Nullable
    public synchronized BerkeleyDBNameSpace getNameSpace(String name, SecondaryKeyProvider provider) {
        BerkeleyDBNameSpace namespace = this.namespaces.get(name);
        if (namespace == null) {
            try {
                this.openEnvironment();
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to open environment " + this, e);
            }
            if (this.environment != null) {
                namespace = new BerkeleyDBNameSpace(name, this, provider);
                this.namespaces.put(name, namespace);
            }
        } else {
            SecondaryKeyProvider oldProvider = namespace.getSecondaryKeyProvider();
            if (provider == null && oldProvider != null || provider != null && oldProvider == null || provider != null && oldProvider != null && !oldProvider.equals(provider)) {
                throw new IllegalArgumentException("NameSpace already opened with a different secondary key provider");
            }
        }
        if (namespace != null) {
            ++namespace.refCount;
        }
        return namespace;
    }

    public synchronized void closeNameSpace(BerkeleyDBNameSpace namespace) {
        if (--namespace.refCount == 0) {
            try {
                namespace.closeDatabase();
                this.namespaces.remove(namespace.getName());
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to close database " + namespace, e);
            }
        }
    }

    public synchronized void deleteNameSpace(final String name) {
        try {
            this.openEnvironment();
        }
        catch (DatabaseException e) {
            PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to open environment " + this, e);
        }
        if (this.environment != null) {
            try {
                this.run(new BerkeleyDBOperation<Void>(){

                    @Override
                    public Void run() throws DatabaseException {
                        BerkeleyDBEnvironment.this.environment.removeDatabase(null, name);
                        return null;
                    }
                });
            }
            catch (DatabaseNotFoundException e) {
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to remove database " + name + " from " + this, e);
            }
        }
    }

    @NotNull
    public synchronized Iterator<String> getNameSpaceIterator(final String prefix, final boolean ignorecase, final boolean reverse) {
        try {
            this.openEnvironment();
        }
        catch (DatabaseException e) {
            PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to open environment " + this, e);
        }
        if (this.environment != null) {
            try {
                return this.run(new BerkeleyDBOperation<Iterator<String>>(){

                    @Override
                    public Iterator<String> run() throws DatabaseException {
                        List dbNamesUnchecked = BerkeleyDBEnvironment.this.environment.getDatabaseNames();
                        ArrayList dbNames = new ArrayList(dbNamesUnchecked);
                        if (reverse) {
                            Collections.reverse(dbNames);
                        }
                        if (prefix.length() == 0) {
                            return dbNames.iterator();
                        }
                        ArrayList<String> names = new ArrayList<String>();
                        if (ignorecase) {
                            int prefixLength = prefix.length();
                            for (String name : dbNames) {
                                if (name.length() < prefixLength || !name.substring(0, prefixLength).equalsIgnoreCase(prefix)) continue;
                                names.add(name);
                            }
                        } else {
                            for (String name : dbNames) {
                                if (!name.startsWith(prefix)) continue;
                                names.add(name);
                            }
                        }
                        names.trimToSize();
                        return names.iterator();
                    }
                });
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.SEVERE, "Unable to iterate databases in " + this, e);
            }
        }
        return EMPTY_ITERATOR;
    }

    public synchronized String toString() {
        if (this.environment != null) {
            try {
                return this.environment.getHome().toString();
            }
            catch (DatabaseException databaseException) {
                // empty catch block
            }
        }
        return this.location;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized <T> T run(BerkeleyDBOperation<T> operation) throws DatabaseException {
        boolean wasInterrupted = Thread.interrupted();
        try {
            T t = operation.run();
            return t;
        }
        catch (RunRecoveryException e) {
            wasInterrupted = Thread.interrupted() || wasInterrupted;
            this.recover(false);
            T t = operation.run();
            return t;
        }
        catch (DatabaseException e) {
            this.recover(true);
            T t = operation.run();
            return t;
        }
        finally {
            if (wasInterrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private synchronized void recover(boolean delete) throws DatabaseException {
        PersistenceLogger.getLogger().fine("Recovering " + this.environment);
        for (BerkeleyDBNameSpace namespace : this.namespaces.values()) {
            try {
                namespace.closeDatabase();
            }
            catch (DatabaseException e) {}
        }
        File home = null;
        if (this.environment != null) {
            home = this.environment.getHome();
        }
        this.closeEnvironment();
        if (delete && home != null) {
            this.deleteEnvironment(home);
        }
        this.openEnvironment();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void openEnvironment() throws DatabaseException {
        if (this.environment != null) {
            try {
                this.environment.compress();
                this.environment.evictMemory();
            }
            catch (DatabaseException e) {
                PersistenceLogger.getLogger().log(Level.WARNING, "Unable to clean up memory", e);
            }
            return;
        }
        Throwable lastException = null;
        for (int i = 0; i < 5; ++i) {
            String dirname = String.format("%08d", i);
            File dir = new File(this.location, dirname);
            if (!dir.exists() && !dir.mkdirs()) {
                PersistenceLogger.getLogger().warning("Cannot write to " + this.location);
                break;
            }
            boolean wasInterrupted = Thread.interrupted();
            OutOfFileHandlesDialog.startTesting();
            try {
                this.environment = new Environment(dir, ENVIRONMENT_CONFIG);
                return;
            }
            catch (RunRecoveryException e) {
                boolean bl = wasInterrupted = Thread.interrupted() || wasInterrupted;
                if (!wasInterrupted) {
                    this.deleteEnvironment(dir);
                }
                try {
                    this.environment = new Environment(dir, ENVIRONMENT_CONFIG);
                    return;
                }
                catch (LogException ee) {
                    if (!OutOfFileHandlesDialog.isOutOfFileHandlesError(ee)) continue;
                    OutOfFileHandlesDialog.show();
                    return;
                }
                catch (DatabaseException ee) {
                    lastException = ee;
                    continue;
                }
            }
            catch (LogException e) {
                if (OutOfFileHandlesDialog.isOutOfFileHandlesError(e)) {
                    OutOfFileHandlesDialog.show();
                    return;
                }
                lastException = e;
                continue;
            }
            catch (DatabaseException e) {
                if (this.deleteEnvironment(dir)) {
                    try {
                        this.environment = new Environment(dir, ENVIRONMENT_CONFIG);
                        return;
                    }
                    catch (DatabaseException ee) {
                        lastException = ee;
                        continue;
                    }
                }
                lastException = e;
                continue;
            }
            finally {
                OutOfFileHandlesDialog.finishTesting();
                if (wasInterrupted) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        if (lastException != null) {
            throw lastException;
        }
    }

    private synchronized void closeEnvironment() throws DatabaseException {
        if (this.environment != null) {
            this.cleanEnvironment(false);
            try {
                this.environment.close();
            }
            catch (DatabaseException databaseException) {
                // empty catch block
            }
            this.environment = null;
        }
    }

    private synchronized boolean deleteEnvironment(File home) {
        PersistenceLogger.getLogger().fine("Deleting " + this);
        File lockFile = new File(home, "je.lck");
        if (lockFile.isFile() && !lockFile.delete()) {
            PersistenceLogger.getLogger().fine("Unable to delete " + lockFile);
            return false;
        }
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".jdb");
            }
        };
        boolean success = true;
        File[] files = home.listFiles(filter);
        if (files != null) {
            for (File file : files) {
                if (!file.isFile() || file.delete()) continue;
                PersistenceLogger.getLogger().fine("Unable to delete " + file);
                success = false;
            }
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void cleanEnvironment(boolean full) {
        if (this.environment != null) {
            PerformanceLogger.get().startTiming("BerkeleyDBEnvironment.cleanEnvironment");
            long cleanLogTime = 0L;
            long checkpointTime = 0L;
            try {
                if (full) {
                    this.environment.compress();
                }
                long cleanLogStart = System.nanoTime();
                boolean cleaned = false;
                while (this.environment.cleanLog() > 0) {
                    cleaned = true;
                    if (full) continue;
                }
                cleanLogTime = (System.nanoTime() - cleanLogStart) / 1000000L;
                if (cleaned) {
                    long checkpointStart = System.nanoTime();
                    CheckpointConfig config = new CheckpointConfig();
                    config.setForce(full);
                    this.environment.checkpoint(config);
                    checkpointTime = (System.nanoTime() - checkpointStart) / 1000000L;
                }
            }
            catch (DatabaseException e) {
            }
            finally {
                PerformanceLogger.get().stopTiming("BerkeleyDBEnvironment.cleanEnvironment", "Cleaned Berkeley DB environment for " + this + " (" + cleanLogTime + "ms cleanLog, " + checkpointTime + "ms checkpoint)", 200);
            }
        }
    }

    @NotNull
    synchronized Database openDatabase(final String name) throws DatabaseException {
        if (this.environment != null) {
            return this.run(new BerkeleyDBOperation<Database>(){

                @Override
                public Database run() throws DatabaseException {
                    return BerkeleyDBEnvironment.this.environment.openDatabase(null, name, DATABASE_CONFIG);
                }
            });
        }
        throw new DatabaseException("Environment " + this + " not open");
    }

    @NotNull
    synchronized SecondaryDatabase openSecondaryDatabase(final String name, final Database primaryDatabase, final SecondaryKeyProvider provider) throws DatabaseException {
        if (this.environment != null) {
            return this.run(new BerkeleyDBOperation<SecondaryDatabase>(){

                @Override
                public SecondaryDatabase run() throws DatabaseException {
                    SecondaryConfig config = new SecondaryConfig();
                    config.setAllowCreate(true);
                    config.setSortedDuplicates(true);
                    config.setMultiKeyCreator((SecondaryMultiKeyCreator)new SecondaryKeyCreatorAdapter(provider, name));
                    return BerkeleyDBEnvironment.this.environment.openSecondaryDatabase(null, name, primaryDatabase, config);
                }
            });
        }
        throw new DatabaseException("Environment " + this + " not open");
    }

    synchronized void flush() throws DatabaseException {
        if (this.environment != null) {
            this.run(new BerkeleyDBOperation<Void>(){

                @Override
                public Void run() throws DatabaseException {
                    BerkeleyDBEnvironment.this.environment.sync();
                    return null;
                }
            });
        }
    }

    static {
        BerkeleyDBEnvironment.initConfig();
    }
}

